function bubble = findBubble(imgName)
	
	config = getConfig(); 
	
	%threshold for long horizontal lines
	th_h = str2num(config.('th_h'));  
	%threshold for short vertical lines
	th_v = str2num(config.('th_v'));  
	%thickness of markers threshold
	th_vv = str2num(config.('th_vv')); 

	%rotate image by degrees
	rotAngle = str2num(config.('rotAngle')); 
	%gaussian
	gauss = str2num(config.('gauss')); 
	%canny
	canny = str2num(config.('canny')); 
	%active contour nr of cycles
	acCycles = str2num(config.('acCycles')); 
	
	
	

	img = imread(imgName);
	[h, w, ch] = size(img); 
	img_orig = imrotate(img, rotAngle);
	img = imgaussfilt(img_orig, gauss); 
	imgg = rgb2gray(img); 
	imge = edge(imgg, 'canny', canny);

	[H,theta,rho] = hough(imge); 
	
	
	%find the longest line -- only one
	peak = houghpeaks(H, 5, 'Threshold', 100);

	%find the angle of all lines perpendicular to longest line -- these are the markers (and other lines)
	if peak(2, 2) < 90
		angle90 = peak(2, 2) + 90; 
	else
		angle90 = peak(2, 2) - 90; 
	end

	angle0 = peak(2, 2);

	%get all lines parallel with the markers
	peaks90 = H(:, angle90);
	peaks0 = H(:, angle0);

	%select the lines above threshold and find the markers around the bubble
	%these markers have the largest distance rho between them, compared with all other distances
	[points90, markerLeft] = getPoints(peaks90, rho, ceil(th_v*h)); 
	[points0, markerTop] = getPoints(peaks0, rho, th_h); 

	[A, B] = findEdges(points90, markerLeft, th_vv, 'back'); 	%right-LEFT and left-LEFT
	[C, D] = findEdges(points90, markerLeft, th_vv, 'fore'); 	%left-RIGHT and right-RIGHT

	%calculate the lines - vertical
	cosAng90 = cosd(theta(angle90)); 
	sinAng90 = sind(theta(angle90)); 

	xs90 = [0 length(img)]; 

	ys1 = [(points90(A, 2) - xs90(1) * cosAng90) / sinAng90, (points90(A, 2) - xs90(2) * cosAng90) / sinAng90]; 
	ys2 = [((points90(C, 2) - (points90(A, 4) - points90(D, 4))) - xs90(1) * cosAng90) / sinAng90, ((points90(C, 2) +15) - xs90(2) * cosAng90) / sinAng90]; 


	%calculate the lines - horizontal
	cosAng0 = cosd(theta(angle0)); 
	sinAng0 = sind(theta(angle0)); 
	xs0 = [0 length(img)]; 

	ys10 = [(points0(markerTop, 2) - xs0(1) * cosAng0) / sinAng0, (points0(markerTop, 2) - xs0(2) * cosAng0) / sinAng0]; 
	ys11 = [(points0(markerTop+1, 2) - xs0(1) * cosAng0) / sinAng0, (points0(markerTop+1, 2) - xs0(2) * cosAng0) / sinAng0]; 



	%draw the lines
	imgg2 = rgb2gray(img_orig); 
	figure('visible', 'off'); 
	imshow(imgg2); 
	hold on; 

	line(xs90, ys1, 'color', 'r', 'lineWidth', 1.5);
	line(xs90, ys2, 'color', 'r', 'lineWidth', 1.5);

	line(xs0, ys10, 'color', 'r', 'lineWidth', 1.5);
	line(xs0, ys11, 'color', 'r', 'lineWidth', 1.5);

	hold off; 



	%get the figure-data (that includes the lines) and create a straight image
	F = getframe; 
	img_lines = F.cdata; 
	img_str = imrotate(img_lines, theta(angle90), 'crop'); 
	%figure; imshow(img_str); 
	close(figure); 

	%create a mask around the bubble location
	[mask, corners] = getMask(img_str);
	%figure; imshow(mask);


	%find the bubble
	bubble_contour = activecontour(img_str, mask, acCycles);
	%imshow(bubble_contour);
	%crop the bubble image - not much use for the bottom portion of image
	bubble = imcrop(bubble_contour, [corners(1,2) (corners(1,1)+5) (corners(4,2)-corners(1,2)) ceil((corners(4,1)-corners(1,1))/2)]); 
end





function [mask, corners] = getMask(img)
	corners = []; 
	[h, w, ch] = size(img); 
	mask = (img(:,:,1) == 255) & (img(:,:,2) == 0) & (img(:,:,3) == 0); 
	topLine = false;
	bottomLine = false;
	
	for i = 1: h
		if all(mask(i))
			mask(i, :) = 0; 
			if bottomLine == true
				topLine = false;
			elseif topLine == false && (all(mask(i+1))) == false
				topLine = true; 
				mask(i+1, :) = 0; 
			end
		else
			if topLine == true
				if (all(mask(i+1))) == true
					bottomLine = true; 
					topLine = false; 
					mask(i, :) = 0;
					corners(3, 1) = i+1; 
					corners(4, 1) = i+1; 
				else
					idx = find(mask(i,:));
					if length(idx) == 0
						continue;
					end
					mask(i, idx(1):idx(length(idx))) = 1;
					if length(corners) == 0
						corners = [corners; [i, idx(1)]]; 
						corners = [corners; [i, idx(length(idx))]];
						corners(3, 2) = idx(1); 
						corners(4, 2) = idx(length(idx)); 
					end
				end
			else
				mask(i, :) = 0; 
			end
		end
	end
	
end




%loop through the lines -- this is equal to the height of the image
% peaks contains all entries in a theta column of the accumulator
function [points, largestDist_id, largestDist] = getPoints(peaks, rho, threshold)
	points = [];
	counter = 1;
	largestDist = 0; 
	largestDist_id = -1; 

	for i = 1:length(peaks)
		dist = -1; 
		if peaks(i) > threshold
			if counter > 1
				prev = points(counter -  1, 1); 
				dist = i - prev; 
				if dist > largestDist
					largestDist = dist;				%save largest distance -- measured from previous index to this one
					largestDist_id = counter - 1; 	% and save its index -- the previous index
				end
			end
			
			points(counter, :) = [i, rho(i), peaks(i), dist];
			counter = counter + 1; 
		end
	end
end



% find better edges, starting from the left-center point, 
% right before the largest distance between bands
function [p1, p2] = findEdges(points, center, length, dir)
	
	if dir == 'back'
		if points(center, 4) >= length
			p1 = center; 
			p2 = center-1; 
		else
			pointNotFound = true; 
			while pointNotFound
				center = center - 1; 
				if points(center, 4) >= length
					p1 = center; 
					p2 = center - 1; 
					pointNotFound = false; 
				end
			end
		end
	else
		p1 = center + 1; 
		
		pointNotFound = true; 
		center =  p1; 
		while pointNotFound
			center =  center + 1; 
			if points(center, 4) >= length
				p2 = center; 
				pointNotFound = false; 
			end
		end
	end
end